<?php
/**
 * Logo Helper Functions
 * Provides easy access to company and event logos throughout the system
 */

require_once 'db_config.php';

/**
 * Get the company logo path from database
 */
function getCompanyLogo() {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'companyLogo'");
        $stmt->execute();
        return $stmt->fetchColumn() ?: null;
    } catch (Exception $e) {
        error_log('Error getting company logo: ' . $e->getMessage());
        return null;
    }
}

/**
 * Get the event logo path from database
 */
function getEventLogo() {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'eventLogo'");
        $stmt->execute();
        return $stmt->fetchColumn() ?: null;
    } catch (Exception $e) {
        error_log('Error getting event logo: ' . $e->getMessage());
        return null;
    }
}

/**
 * Get the full URL to the company logo
 */
function getCompanyLogoUrl($baseUrl = '') {
    $logoPath = getCompanyLogo();
    if (!$logoPath) {
        return null;
    }
    
    // If no base URL provided, try to determine it
    if (empty($baseUrl)) {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // Get the correct base path - remove /admin/ if present in script name
        $scriptDir = dirname($_SERVER['SCRIPT_NAME']);
        if (strpos($scriptDir, '/admin') !== false) {
            $scriptDir = str_replace('/admin', '', $scriptDir);
        }
        $baseUrl = $protocol . '://' . $host . $scriptDir;
    }
    
    return rtrim($baseUrl, '/') . '/' . ltrim($logoPath, '/');
}

/**
 * Get the full URL to the event logo
 */
function getEventLogoUrl($baseUrl = '') {
    $logoPath = getEventLogo();
    if (!$logoPath) {
        return null;
    }
    
    // If no base URL provided, try to determine it
    if (empty($baseUrl)) {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // Get the correct base path - remove /admin/ if present in script name
        $scriptDir = dirname($_SERVER['SCRIPT_NAME']);
        if (strpos($scriptDir, '/admin') !== false) {
            $scriptDir = str_replace('/admin', '', $scriptDir);
        }
        $baseUrl = $protocol . '://' . $host . $scriptDir;
    }
    
    return rtrim($baseUrl, '/') . '/' . ltrim($logoPath, '/');
}

/**
 * Get company logo HTML img tag for use in templates
 */
function getCompanyLogoHtml($maxWidth = '200px', $maxHeight = '100px', $alt = 'Company Logo', $class = '') {
    $logoPath = getCompanyLogo();
    if (!$logoPath) {
        return '';
    }
    
    // Fix path for admin pages - remove admin/ prefix if present
    $displayPath = $logoPath;
    if (strpos($_SERVER['REQUEST_URI'], '/admin/') !== false) {
        $displayPath = '../' . $logoPath;
    }
    
    $style = "max-width: {$maxWidth}; max-height: {$maxHeight}; height: auto;";
    $classAttr = $class ? " class=\"{$class}\"" : '';
    
    return "<img src=\"{$displayPath}\" alt=\"{$alt}\" style=\"{$style}\"{$classAttr}>";
}

/**
 * Get event logo HTML img tag for use in templates
 */
function getEventLogoHtml($maxWidth = '200px', $maxHeight = '100px', $alt = 'Event Logo', $class = '') {
    $logoPath = getEventLogo();
    if (!$logoPath) {
        return '';
    }
    
    // Fix path for admin pages - remove admin/ prefix if present
    $displayPath = $logoPath;
    if (strpos($_SERVER['REQUEST_URI'], '/admin/') !== false) {
        $displayPath = '../' . $logoPath;
    }
    
    $style = "max-width: {$maxWidth}; max-height: {$maxHeight}; height: auto;";
    $classAttr = $class ? " class=\"{$class}\"" : '';
    
    return "<img src=\"{$displayPath}\" alt=\"{$alt}\" style=\"{$style}\"{$classAttr}>";
}

/**
 * Get company logo for email templates (with full URL)
 */
function getCompanyLogoForEmail($maxWidth = '200px', $maxHeight = '100px', $alt = 'Company Logo') {
    $logoUrl = getCompanyLogoUrl();
    if (!$logoUrl) {
        return '';
    }
    
    $style = "max-width: {$maxWidth}; max-height: {$maxHeight}; height: auto; display: block; margin: 0 auto;";
    
    return "<img src=\"{$logoUrl}\" alt=\"{$alt}\" style=\"{$style}\">";
}

/**
 * Get event logo for email templates (with full URL)
 */
function getEventLogoForEmail($maxWidth = '200px', $maxHeight = '100px', $alt = 'Event Logo') {
    $logoUrl = getEventLogoUrl();
    if (!$logoUrl) {
        return '';
    }
    
    $style = "max-width: {$maxWidth}; max-height: {$maxHeight}; height: auto; display: block; margin: 0 auto;";
    
    return "<img src=\"{$logoUrl}\" alt=\"{$alt}\" style=\"{$style}\">";
}

/**
 * Check if company logo exists
 */
function hasCompanyLogo() {
    $logoPath = getCompanyLogo();
    return $logoPath && file_exists($logoPath);
}

/**
 * Check if event logo exists
 */
function hasEventLogo() {
    $logoPath = getEventLogo();
    return $logoPath && file_exists($logoPath);
}

/**
 * Get company logo dimensions if available
 */
function getCompanyLogoDimensions() {
    $logoPath = getCompanyLogo();
    if (!$logoPath || !file_exists($logoPath)) {
        return null;
    }
    
    $imageInfo = getimagesize($logoPath);
    if ($imageInfo) {
        return [
            'width' => $imageInfo[0],
            'height' => $imageInfo[1],
            'type' => $imageInfo[2],
            'mime' => $imageInfo['mime']
        ];
    }
    
    return null;
}

/**
 * Get event logo dimensions if available
 */
function getEventLogoDimensions() {
    $logoPath = getEventLogo();
    if (!$logoPath || !file_exists($logoPath)) {
        return null;
    }
    
    $imageInfo = getimagesize($logoPath);
    if ($imageInfo) {
        return [
            'width' => $imageInfo[0],
            'height' => $imageInfo[1],
            'type' => $imageInfo[2],
            'mime' => $imageInfo['mime']
        ];
    }
    
    return null;
}

/**
 * Get the appropriate logo for admin interface (company logo preferred, fallback to event logo)
 */
function getAdminLogo() {
    $companyLogo = getCompanyLogo();
    if ($companyLogo) {
        return $companyLogo;
    }
    return getEventLogo();
}

/**
 * Get admin logo HTML for navigation
 */
function getAdminLogoHtml($maxWidth = '40px', $maxHeight = '40px', $alt = 'Admin Logo', $class = 'admin-logo-img') {
    $logoPath = getAdminLogo();
    if (!$logoPath) {
        return '';
    }
    
    // Fix path for admin pages - remove admin/ prefix if present
    $displayPath = $logoPath;
    if (strpos($_SERVER['REQUEST_URI'], '/admin/') !== false) {
        $displayPath = '../' . $logoPath;
    }
    
    $style = "max-width: {$maxWidth}; max-height: {$maxHeight}; height: auto;";
    $classAttr = $class ? " class=\"{$class}\"" : '';
    
    return "<img src=\"{$displayPath}\" alt=\"{$alt}\" style=\"{$style}\"{$classAttr}>";
}
?>
