<?php
/**
 * Environment Diagnostic Script
 * Upload this to your live server to diagnose environment loading issues
 */

echo "🔍 Environment Diagnostic Report\n";
echo "================================\n";
echo "🔒 SECURE CONFIGURATION - No Hardcoded Fallbacks\n\n";

// Check if .env file exists
$envPath = __DIR__ . '/.env';
echo "1. Environment File Check:\n";
echo "   Path: " . $envPath . "\n";
echo "   Exists: " . (file_exists($envPath) ? "✅ YES" : "❌ NO") . "\n";
if (file_exists($envPath)) {
    echo "   Readable: " . (is_readable($envPath) ? "✅ YES" : "❌ NO") . "\n";
    echo "   Size: " . filesize($envPath) . " bytes\n";
}
echo "\n";

// Test environment loader
echo "2. Environment Loader Test:\n";
try {
    require_once 'env_loader.php';
    echo "   Loader: ✅ LOADED\n";
    
    // Test specific variables
    $testVars = ['DB_HOST', 'DB_NAME', 'DB_USER', 'DB_PASS', 'STRIPE_PUBLISHABLE_KEY', 'STRIPE_SECRET_KEY'];
    foreach ($testVars as $var) {
        $value = EnvLoader::get($var);
        echo "   {$var}: " . ($value ? "✅ SET (" . strlen($value) . " chars)" : "❌ NOT SET") . "\n";
    }
} catch (Exception $e) {
    echo "   Loader: ❌ FAILED - " . $e->getMessage() . "\n";
}
echo "\n";

// Test Stripe configuration
echo "3. Stripe Configuration Test:\n";
try {
    require_once 'stripe_config.php';
    
    $pubKey = StripeConfig::getPublishableKey();
    $secretKey = StripeConfig::getSecretKey();
    
    echo "   Publishable Key: " . ($pubKey ? "✅ SET (" . strlen($pubKey) . " chars)" : "❌ NOT SET") . "\n";
    echo "   Secret Key: " . ($secretKey ? "✅ SET (" . strlen($secretKey) . " chars)" : "❌ NOT SET") . "\n";
    
    // Test if keys are valid format
    if ($pubKey) {
        echo "   Pub Key Format: " . (strpos($pubKey, 'pk_') === 0 ? "✅ VALID" : "❌ INVALID") . "\n";
    }
    if ($secretKey) {
        echo "   Secret Key Format: " . (strpos($secretKey, 'sk_') === 0 ? "✅ VALID" : "❌ INVALID") . "\n";
    }
    
} catch (Exception $e) {
    echo "   Stripe Config: ❌ FAILED - " . $e->getMessage() . "\n";
}
echo "\n";

// Test database connection
echo "4. Database Connection Test:\n";
try {
    require_once 'db_config.php';
    $pdo = getDBConnection();
    echo "   Connection: ✅ SUCCESS\n";
} catch (Exception $e) {
    echo "   Connection: ❌ FAILED - " . $e->getMessage() . "\n";
}
echo "\n";

// System information
echo "5. System Information:\n";
echo "   PHP Version: " . phpversion() . "\n";
echo "   Working Directory: " . getcwd() . "\n";
echo "   Script Path: " . __FILE__ . "\n";
echo "   User: " . (function_exists('posix_getpwuid') ? posix_getpwuid(posix_geteuid())['name'] : 'unknown') . "\n";

// File permissions
echo "\n6. File Permissions:\n";
$files = ['.env', 'env_loader.php', 'db_config.php', 'stripe_config.php'];
foreach ($files as $file) {
    if (file_exists($file)) {
        echo "   {$file}: " . substr(sprintf('%o', fileperms($file)), -4) . "\n";
    }
}

echo "\n" . str_repeat("=", 50) . "\n";
echo "Diagnostic completed. Please review the results above.\n";
?>
