<?php
require_once 'auth_rbac.php';
require_once '../db_config.php';

require_once 'config_utils.php';

// Require admin login
requirePermission('settings', 'view');

$currentPage = 'settings';

$message = '';
$error = '';

// Handle success messages from redirects
if (isset($_GET['updated'])) {
    switch ($_GET['updated']) {
        case '1':
            $message = 'Setting updated successfully';
            break;
        case 'dietary':
            $message = 'Dietary options updated successfully';
            break;
        case 'terms':
            $message = 'Terms and conditions updated successfully';
            break;
        case 'email':
            $message = 'Email settings updated successfully';
            break;
        case 'logo':
            $message = 'Company logo uploaded successfully!';
            break;
        case 'event_logo':
            $message = 'Event logo uploaded successfully!';
            break;
        case 'webhook':
            $message = 'Stripe webhook secret updated successfully!';
            break;
    }
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo = getDBConnection();
        
        if (isset($_POST['update_setting'])) {
            $key = $_POST['setting_key'];
            $value = $_POST['setting_value'];
            $type = $_POST['setting_type'] ?? 'text';
            $description = $_POST['setting_description'] ?? '';
            
            $stmt = $pdo->prepare("
                INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) 
                VALUES (?, ?, ?, ?) 
                ON DUPLICATE KEY UPDATE 
                    setting_value = VALUES(setting_value),
                    updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute([$key, $value, $type, $description]);
            
            // Special handling for webhook secret
            if ($key === 'stripeWebhookSecret') {
                // Validate webhook secret format
                if (strpos($value, 'whsec_') === 0) {
                    header('Location: settings.php?updated=webhook');
                } else {
                    $error = 'Warning: Webhook secret should start with "whsec_" for proper Stripe integration.';
                }
            } else {
                // Redirect to prevent form resubmission
                header('Location: settings.php?updated=1');
            }
            
            if (!$error) {
                exit;
            }
            
        } elseif (isset($_POST['update_dietary_options'])) {
            $dietary_options_text = $_POST['dietary_options'] ?? '';
            
            // Convert textarea input (one per line) to array
            $dietary_options_array = array_filter(
                array_map('trim', explode("\n", $dietary_options_text)),
                function($option) { return !empty($option); }
            );
            
            // Convert to JSON format for database storage
            $dietary_options_json = json_encode($dietary_options_array);
            
            $stmt = $pdo->prepare("
                INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) 
                VALUES (?, ?, ?, ?) 
                ON DUPLICATE KEY UPDATE 
                    setting_value = VALUES(setting_value),
                    updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute(['dietaryOptions', $dietary_options_json, 'json', 'Available dietary requirement options']);
            
            // Redirect to prevent form resubmission
            header('Location: settings.php?updated=dietary');
            exit;
            
        } elseif (isset($_POST['update_terms'])) {
            $terms_conditions = $_POST['terms_conditions'] ?? '';
            
            $stmt = $pdo->prepare("
                INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) 
                VALUES (?, ?, ?, ?) 
                ON DUPLICATE KEY UPDATE 
                    setting_value = VALUES(setting_value),
                    updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute(['invoiceTermsConditions', $terms_conditions, 'text', 'Terms and conditions for invoices']);
            
            // Redirect to prevent form resubmission
            header('Location: settings.php?updated=terms');
            exit;
            
        } elseif (isset($_POST['update_email_settings'])) {
            // Handle email settings update
            $emailSettings = [
                'brevoApiKey' => $_POST['brevo_api_key'] ?? '',
                'brevoSenderEmail' => $_POST['brevo_sender_email'] ?? 'noreply@xchange2025.com',
                'brevoSenderName' => $_POST['brevo_sender_name'] ?? 'XChange 2025 Team',
                'adminNotificationEmails' => $_POST['admin_notification_emails'] ?? '',
                'bankTransferDetails' => $_POST['bank_transfer_details'] ?? ''
            ];
            
            $stmt = $pdo->prepare("
                INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) 
                VALUES (?, ?, ?, ?) 
                ON DUPLICATE KEY UPDATE 
                    setting_value = VALUES(setting_value),
                    updated_at = CURRENT_TIMESTAMP
            ");
            
            $descriptions = [
                'brevoApiKey' => 'Brevo API Key for email sending',
                'brevoSenderEmail' => 'Sender email address for outgoing emails',
                'brevoSenderName' => 'Sender name for outgoing emails',
                'adminNotificationEmails' => 'Admin notification email addresses (comma-separated)',
                'bankTransferDetails' => 'Bank transfer instructions for payment emails'
            ];
            
            foreach ($emailSettings as $key => $value) {
                $type = ($key === 'brevoApiKey') ? 'string' : 'text';
                $stmt->execute([$key, $value, $type, $descriptions[$key]]);
            }
            
            // Redirect to prevent form resubmission
            header('Location: settings.php?updated=email');
            exit;
            
        } elseif (isset($_POST['upload_company_logo'])) {
            // Handle company logo upload
            if (isset($_FILES['company_logo']) && $_FILES['company_logo']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = '../uploads/';
                if (!is_dir($uploadDir)) {
                    mkdir($uploadDir, 0755, true);
                }
                
                $file = $_FILES['company_logo'];
                $fileName = $file['name'];
                $fileTmpName = $file['tmp_name'];
                $fileSize = $file['size'];
                $fileError = $file['error'];
                $fileType = $file['type'];
                
                // Get file extension
                $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                $allowedExts = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'webp'];
                
                if (in_array($fileExt, $allowedExts)) {
                    if ($fileSize < 5000000) { // 5MB limit
                        // Generate unique filename
                        $newFileName = 'company_logo_' . time() . '.' . $fileExt;
                        $uploadPath = $uploadDir . $newFileName;
                        
                        if (move_uploaded_file($fileTmpName, $uploadPath)) {
                            // Delete old logo if exists
                            $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'companyLogo'");
                            $stmt->execute();
                            $oldLogo = $stmt->fetchColumn();
                            if ($oldLogo && file_exists('../' . $oldLogo)) {
                                unlink('../' . $oldLogo);
                            }
                            
                            // Save new logo path to database
                            $logoPath = 'uploads/' . $newFileName;
                            $stmt = $pdo->prepare("
                                INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) 
                                VALUES (?, ?, ?, ?) 
                                ON DUPLICATE KEY UPDATE 
                                    setting_value = VALUES(setting_value),
                                    updated_at = CURRENT_TIMESTAMP
                            ");
                            $stmt->execute(['companyLogo', $logoPath, 'file', 'Company/Organization logo image']);
                            
                            // Redirect to prevent form resubmission
                            header('Location: settings.php?updated=logo');
                            exit;
                        } else {
                            $error = 'Failed to upload company logo. Please try again.';
                        }
                    } else {
                        $error = 'Company logo file is too large. Maximum size is 5MB.';
                    }
                } else {
                    $error = 'Invalid file type. Please upload JPG, PNG, GIF, SVG, or WebP files only.';
                }
            } else {
                $error = 'Please select a company logo file to upload.';
            }
            
        } elseif (isset($_POST['upload_event_logo'])) {
            // Handle event logo upload
            if (isset($_FILES['event_logo']) && $_FILES['event_logo']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = '../uploads/';
                if (!is_dir($uploadDir)) {
                    mkdir($uploadDir, 0755, true);
                }
                
                $file = $_FILES['event_logo'];
                $fileName = $file['name'];
                $fileTmpName = $file['tmp_name'];
                $fileSize = $file['size'];
                $fileError = $file['error'];
                $fileType = $file['type'];
                
                // Get file extension
                $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                $allowedExts = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'webp'];
                
                if (in_array($fileExt, $allowedExts)) {
                    if ($fileSize < 5000000) { // 5MB limit
                        // Generate unique filename
                        $newFileName = 'event_logo_' . time() . '.' . $fileExt;
                        $uploadPath = $uploadDir . $newFileName;
                        
                        if (move_uploaded_file($fileTmpName, $uploadPath)) {
                            // Delete old logo if exists
                            $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'eventLogo'");
                            $stmt->execute();
                            $oldLogo = $stmt->fetchColumn();
                            if ($oldLogo && file_exists('../' . $oldLogo)) {
                                unlink('../' . $oldLogo);
                            }
                            
                            // Save new logo path to database
                            $logoPath = 'uploads/' . $newFileName;
                            $stmt = $pdo->prepare("
                                INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) 
                                VALUES (?, ?, ?, ?) 
                                ON DUPLICATE KEY UPDATE 
                                    setting_value = VALUES(setting_value),
                                    updated_at = CURRENT_TIMESTAMP
                            ");
                            $stmt->execute(['eventLogo', $logoPath, 'file', 'Event logo image']);
                            
                            // Redirect to prevent form resubmission
                            header('Location: settings.php?updated=event_logo');
                            exit;
                        } else {
                            $error = 'Failed to upload logo. Please try again.';
                        }
                    } else {
                        $error = 'Logo file is too large. Maximum size is 5MB.';
                    }
                } else {
                    $error = 'Invalid file type. Please upload JPG, PNG, GIF, SVG, or WebP files only.';
                }
            } else {
                $error = 'Please select a logo file to upload.';
            }
            
        } elseif (isset($_POST['remove_event_logo'])) {
            // Handle event logo removal
            $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'eventLogo'");
            $stmt->execute();
            $logoPath = $stmt->fetchColumn();
            
            if ($logoPath) {
                // Delete file if exists
                if (file_exists('../' . $logoPath)) {
                    unlink('../' . $logoPath);
                }
                
                // Remove from database
                $stmt = $pdo->prepare("DELETE FROM admin_settings WHERE setting_key = 'eventLogo'");
                $stmt->execute();
                
                $message = 'Event logo removed successfully!';
            } else {
                $error = 'No event logo found to remove.';
            }
            
        } elseif (isset($_POST['remove_company_logo'])) {
            // Handle company logo removal
            $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'companyLogo'");
            $stmt->execute();
            $logoPath = $stmt->fetchColumn();
            
            if ($logoPath) {
                // Delete file if exists
                if (file_exists('../' . $logoPath)) {
                    unlink('../' . $logoPath);
                }
                
                // Remove from database
                $stmt = $pdo->prepare("DELETE FROM admin_settings WHERE setting_key = 'companyLogo'");
                $stmt->execute();
                
                $message = 'Company logo removed successfully!';
            } else {
                $error = 'No company logo found to remove.';
            }
            
        } elseif (isset($_POST['reset_to_defaults'])) {
            // Reset all settings to default values
            $pdo->beginTransaction();
            $pdo->exec("DELETE FROM admin_settings");
            
            $stmt = $pdo->prepare("INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) VALUES (?, ?, ?, ?)");
            
            // Default settings
            $defaults = [
                ['earlyBirdActive', 'true', 'boolean', 'Enable Early Bird pricing'],
                ['early_bird_enabled', 'enabled', 'string', 'Early bird status (enabled/disabled)'],
                ['earlyBirdDeadline', '2025-08-31T23:59:59', 'string', 'Early bird registration deadline'],
                ['earlyBirdPrice', '500', 'number', 'Early Bird delegate pass price'],
                ['standardPrice', '550', 'number', 'Standard delegate pass price'],
                ['standardTablePrice', '550', 'number', 'Standard table presentation price'],
                ['doubleTablePrice', '850', 'number', 'Double table presentation price'],
                ['maxDelegates', '50', 'number', 'Maximum delegates per registration'],
                ['maxTables', '100', 'number', 'Maximum table bookings allowed'],
                ['eventName', 'AppliedHE Xchange 2025', 'text', 'Event name'],
                ['eventDates', '27–28 November 2025', 'text', 'Event dates'],
                ['eventLocation', 'Sunway University, Malaysia', 'text', 'Event location'],
                ['bankName', 'DBS Bank Ltd Singapore', 'text', 'Bank name for transfers'],
                ['accountName', 'AppliedHE Pte. Ltd.', 'text', 'Bank account name'],
                ['accountNumber', '1234567890', 'text', 'Bank account number'],
                ['swiftCode', 'DBSSSGSG', 'text', 'Bank SWIFT code'],
                ['referencePrefix', 'XCHANGE2025', 'text', 'Payment reference prefix'],
                ['dietaryOptions', '["None","Vegetarian","Vegan","Halal","Kosher","Gluten-free","Dairy-free","Nut allergy","Other"]', 'json', 'Available dietary requirement options'],
                ['invoiceTermsConditions', "• Payment is due within 30 days of invoice date\n• Please include the reference number with your payment\n• For any queries, contact us at finance@appliedhe.com\n• All prices are in USD and subject to change without notice\n• Registration is confirmed upon receipt of payment", 'text', 'Terms and conditions for invoices'],
                ['eventPageUrl', '../index.html', 'text', 'URL for Return to Event Page button'],
                ['systemName', 'XChange 2025 Registration System', 'text', 'System name'],
                ['adminEmail', 'admin@xchange2025.com', 'text', 'System administrator email'],
                ['timezone', 'Asia/Kuala_Lumpur', 'text', 'System timezone'],
                ['currency', 'USD', 'text', 'Default currency'],
                // Brevo Email Settings
                ['brevoApiKey', '', 'string', 'Brevo API Key'],
                ['brevoSenderEmail', 'noreply@xchange2025.com', 'email', 'Brevo sender email address'],
                ['brevoSenderName', 'XChange 2025', 'text', 'Brevo sender name'],
                ['adminNotificationEmails', 'admin@xchange2025.com', 'text', 'Admin notification email addresses (comma-separated)'],
                ['bankTransferDetails', "Bank Name: DBS Bank Ltd Singapore\nAccount Name: AppliedHE Pte. Ltd.\nAccount Number: 1234567890\nSWIFT Code: DBSSSGSG\n\nPlease include your registration number as the payment reference.", 'textarea', 'Bank transfer instructions for emails']
            ];
            
            foreach ($defaults as $default) {
                $stmt->execute($default);
            }
            
            $pdo->commit();
            $message = 'All settings reset to default values successfully';
        }
    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $error = 'Error: ' . $e->getMessage();
    }
}

// Load configuration after POST processing to get latest values
if (!isset($config)) {
    $config = getConfiguration();
}

try {
    $pdo = getDBConnection();
    $stmt = $pdo->query("SELECT COUNT(*) FROM admin_settings");
    $db_settings_count = $stmt->fetchColumn();
    $stmt = $pdo->query("SELECT * FROM admin_settings ORDER BY setting_key");
    $db_settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $db_settings_count = 0;
    $db_settings = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - XChange Admin</title>
    <?php include '../includes/favicon.php'; ?>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../styles.css">
    <link rel="stylesheet" href="admin.css">
</head>
<body>
    <?php include 'includes/admin_nav.php'; ?>

    <div class="admin-content">
        <?php if ($message): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <!-- Settings Tabs -->
        <div class="settings-tabs">
            <button class="tab-button active" onclick="showTab('general')">General Settings</button>
            <button class="tab-button" onclick="showTab('organization')">Organization Settings</button>
            <button class="tab-button" onclick="showTab('pricing')">Pricing & Limits</button>
            <button class="tab-button" onclick="showTab('email')">Email Settings</button>
            <button class="tab-button" onclick="showTab('system')">System Management</button>
        </div>
        
        <div class="tab-content">
            <!-- General Settings Tab -->
            <div id="general" class="tab-pane active">
                <h2 style="margin: 0 0 24px 0;">General Settings</h2>
                
                <div class="settings-grid">
                    <!-- Event Branding -->
                    <div class="settings-card">
                        <h3>🎨 Event Branding</h3>
                        <p style="margin: 0 0 20px 0; color: #64748b; font-size: 14px;">
                            Manage your organization and event logos for consistent branding across the system
                        </p>
                        
                        <?php
                        // Get current logos
                        $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'companyLogo'");
                        $stmt->execute();
                        $currentCompanyLogo = $stmt->fetchColumn();
                        
                        $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'eventLogo'");
                        $stmt->execute();
                        $currentEventLogo = $stmt->fetchColumn();
                        ?>
                        
                        <!-- Company Logo Section -->
                        <div class="setting-row" style="margin-bottom: 30px; padding-bottom: 25px; border-bottom: 1px solid #e2e8f0;">
                            <div class="setting-info">
                                <h4>🏢 Company/Organization Logo</h4>
                                <p>Your organization's logo for official documents, invoices, and admin interface</p>
                                <p style="margin: 8px 0 0 0; font-size: 14px; color: #64748b;">
                                    Supported formats: JPG, PNG, GIF, SVG, WebP (Max: 5MB)
                                </p>
                            </div>
                            
                            <div class="logo-upload-section">
                                <?php if ($currentCompanyLogo): ?>
                                <div class="current-logo">
                                    <img src="../<?php echo htmlspecialchars($currentCompanyLogo); ?>" 
                                         alt="Current Company Logo" 
                                         style="max-width: 200px; max-height: 100px; border: 1px solid #e2e8f0; border-radius: 8px; padding: 10px; background: white;">
                                    <div style="margin-top: 10px;">
                                        <form method="POST" style="display: inline;">
                                            <button type="submit" name="remove_company_logo" class="btn btn-danger btn-sm" 
                                                    onclick="return confirm('Are you sure you want to remove the company logo?')">
                                                🗑️ Remove Company Logo
                                            </button>
                                        </form>
                                    </div>
                                </div>
                                <?php else: ?>
                                <div class="no-logo">
                                    <div style="padding: 20px; border: 2px dashed #d1d5db; border-radius: 8px; text-align: center; color: #6b7280;">
                                        <p style="margin: 0; font-size: 14px;">No company logo uploaded</p>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <form method="POST" enctype="multipart/form-data" style="margin-top: 15px;">
                                    <div class="form-group">
                                        <input type="file" name="company_logo" accept="image/*" class="form-control" 
                                               style="margin-bottom: 10px;">
                                        <button type="submit" name="upload_company_logo" class="btn btn-primary">
                                            📤 Upload Company Logo
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                        
                        <!-- Event Logo Section -->
                        <div class="setting-row">
                            <div class="setting-info">
                                <h4>🎪 Event Logo</h4>
                                <p>Specific logo for this event, displayed in emails and event-related communications</p>
                                <p style="margin: 8px 0 0 0; font-size: 14px; color: #64748b;">
                                    Supported formats: JPG, PNG, GIF, SVG, WebP (Max: 5MB)
                                </p>
                            </div>
                            
                            <div class="logo-upload-section">
                                <?php if ($currentEventLogo): ?>
                                <div class="current-logo">
                                    <img src="../<?php echo htmlspecialchars($currentEventLogo); ?>" 
                                         alt="Current Event Logo" 
                                         style="max-width: 200px; max-height: 100px; border: 1px solid #e2e8f0; border-radius: 8px; padding: 10px; background: white;">
                                    <div style="margin-top: 10px;">
                                        <form method="POST" style="display: inline;">
                                            <button type="submit" name="remove_event_logo" class="btn btn-danger btn-sm" 
                                                    onclick="return confirm('Are you sure you want to remove the event logo?')">
                                                🗑️ Remove Event Logo
                                            </button>
                                        </form>
                                    </div>
                                </div>
                                <?php else: ?>
                                <div class="no-logo">
                                    <div style="padding: 20px; border: 2px dashed #d1d5db; border-radius: 8px; text-align: center; color: #6b7280;">
                                        <p style="margin: 0; font-size: 14px;">No event logo uploaded</p>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <form method="POST" enctype="multipart/form-data" style="margin-top: 15px;">
                                    <div class="form-group">
                                        <input type="file" name="event_logo" accept="image/*" class="form-control" 
                                               style="margin-bottom: 10px;">
                                        <button type="submit" name="upload_event_logo" class="btn btn-primary">
                                            📤 Upload Event Logo
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    
                    <!-- System Status -->
                    <div class="settings-card">
                        <h3>System Status</h3>
                        
                        <div class="setting-row">
                            <div class="setting-info">
                                <h4>Database Connection</h4>
                                <p>Database connectivity status</p>
                            </div>
                            <span class="status-indicator status-active">
                                ✅ Connected
                            </span>
                        </div>
                        
                        <div class="setting-row">
                            <div class="setting-info">
                                <h4>Configuration Source</h4>
                                <p>Database-only configuration active</p>
                            </div>
                            <span class="status-indicator status-active">
                                🗄️ Database Only
                            </span>
                        </div>
                        
                        <div class="setting-row">
                            <div class="setting-info">
                                <h4>Settings Count</h4>
                                <p>Number of settings in database</p>
                            </div>
                            <span class="status-indicator status-active">
                                <?php echo number_format($db_settings_count); ?> entries
                            </span>
                        </div>
                    </div>
                    
                    <!-- Event Page Configuration -->
                    <div class="settings-card">
                        <h3>Event Page Configuration</h3>
                        
                        <form method="POST">
                            <div style="margin-bottom: 20px;">
                                <h4 style="margin: 0 0 8px 0;">Return to Event Page URL</h4>
                                <p style="margin: 0 0 12px 0; color: #64748b;">URL where users are redirected when clicking "Return to Event Page" button after registration/payment. Can be a full URL to any domain.</p>
                                <input type="url" name="setting_value" value="<?php 
                                    $eventPageUrl = '../index.html';
                                    if (isset($db_settings)) {
                                        foreach ($db_settings as $setting) {
                                            if ($setting['setting_key'] === 'eventPageUrl') {
                                                $eventPageUrl = $setting['setting_value'];
                                                break;
                                            }
                                        }
                                    }
                                    echo htmlspecialchars($eventPageUrl);
                                ?>" class="form-input" placeholder="https://appliedhe.com/xchange2025" style="width: 100%; box-sizing: border-box;" required>
                                <input type="hidden" name="setting_key" value="eventPageUrl">
                                <input type="hidden" name="setting_type" value="text">
                                <input type="hidden" name="setting_description" value="URL for Return to Event Page button">
                                <button type="submit" name="update_setting" class="btn" style="margin-top: 12px;">Save</button>
                            </div>
                        </form>
                        
                        <div style="margin-top: 12px; padding: 15px; background: #f0f9ff; border-radius: 6px; border-left: 3px solid #0ea5e9;">
                            <p style="margin: 0 0 8px 0; font-size: 14px; color: #0c4a6e;">
                                <strong>💡 Examples:</strong>
                            </p>
                            <ul style="margin: 0; padding-left: 16px; font-size: 14px; color: #0c4a6e;">
                                <li><strong>Same domain:</strong> ../index.html or /event/main.html</li>
                                <li><strong>Different domain:</strong> https://appliedhe.com/xchange2025</li>
                                <li><strong>External site:</strong> https://yourcompany.com/events</li>
                            </ul>
                        </div>
                    </div>
                    
                    <!-- Stripe Webhook Configuration -->
                    <div class="settings-card">
                        <h3>🔐 Stripe Webhook Secret</h3>
                        <p style="margin: 0 0 20px 0; color: #64748b; font-size: 14px;">
                            Configure your Stripe webhook signing secret for secure payment processing. This secret is used to verify that webhook requests come from Stripe.
                        </p>
                        
                        <?php
                        // Get current webhook secret
                        $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = 'stripeWebhookSecret'");
                        $stmt->execute();
                        $currentWebhookSecret = $stmt->fetchColumn();
                        ?>
                        
                        <form method="POST">
                            <div style="margin-bottom: 20px;">
                                <h4 style="margin: 0 0 8px 0;">Webhook Signing Secret</h4>
                                <p style="margin: 0 0 12px 0; color: #64748b; font-size: 14px;">
                                    Get this from your Stripe Dashboard → Developers → Webhooks → [Your Webhook] → Signing Secret
                                </p>
                                
                                <!-- Current Secret Display (Masked) -->
                                <?php if ($currentWebhookSecret): ?>
                                <div style="margin-bottom: 15px; padding: 12px; background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 6px;">
                                    <div style="display: flex; justify-content: space-between; align-items: center;">
                                        <div>
                                            <strong style="font-size: 14px; color: #374151;">Current Secret:</strong>
                                            <span id="webhook-secret-display" style="font-family: monospace; font-size: 14px; color: #6b7280;">
                                                whsec_••••••••••••••••••••••••••••••••••••••••••••••••••••••••••••
                                            </span>
                                            <span id="webhook-secret-full" style="font-family: monospace; font-size: 14px; color: #6b7280; display: none;">
                                                <?php echo htmlspecialchars($currentWebhookSecret); ?>
                                            </span>
                                        </div>
                                        <button type="button" id="toggle-secret" onclick="toggleWebhookSecret()" 
                                                style="background: none; border: 1px solid #d1d5db; padding: 4px 8px; border-radius: 4px; cursor: pointer; font-size: 12px;">
                                            👁️ Show
                                        </button>
                                    </div>
                                    
                                    <!-- Status Indicator -->
                                    <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #e2e8f0;">
                                        <?php if (strpos($currentWebhookSecret, 'whsec_') === 0): ?>
                                            <span style="color: #059669; font-size: 14px;">✅ Valid webhook secret format</span>
                                        <?php else: ?>
                                            <span style="color: #dc2626; font-size: 14px;">⚠️ Invalid format - should start with 'whsec_'</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php else: ?>
                                <div style="margin-bottom: 15px; padding: 12px; background: #fef3c7; border: 1px solid #f59e0b; border-radius: 6px;">
                                    <span style="color: #92400e; font-size: 14px;">⚠️ No webhook secret configured - webhook requests will fail!</span>
                                </div>
                                <?php endif; ?>
                                
                                <input type="password" name="setting_value" 
                                       value="<?php echo htmlspecialchars($currentWebhookSecret ?? ''); ?>" 
                                       class="form-input" 
                                       placeholder="whsec_1234567890abcdef..." 
                                       style="width: 100%; box-sizing: border-box; font-family: monospace;">
                                       
                                <input type="hidden" name="setting_key" value="stripeWebhookSecret">
                                <input type="hidden" name="setting_type" value="string">
                                <input type="hidden" name="setting_description" value="Stripe webhook signing secret">
                                <button type="submit" name="update_setting" class="btn" style="margin-top: 12px;">
                                    🔄 Update Webhook Secret
                                </button>
                            </div>
                        </form>
                        
                        <div style="margin-top: 20px; padding: 15px; background: #f0f9ff; border-radius: 6px; border-left: 3px solid #0ea5e9;">
                            <p style="margin: 0 0 8px 0; font-size: 14px; color: #0c4a6e;">
                                <strong>📋 How to get your webhook secret:</strong>
                            </p>
                            <ol style="margin: 0; padding-left: 16px; font-size: 14px; color: #0c4a6e;">
                                <li>Go to <strong>Stripe Dashboard</strong> → <strong>Developers</strong> → <strong>Webhooks</strong></li>
                                <li>Click on your webhook endpoint: <code>https://apps.appliedhe.one/xchange_reg/stripe_webhook.php</code></li>
                                <li>Click <strong>"Reveal"</strong> next to "Signing secret"</li>
                                <li>Copy the secret (starts with <code>whsec_</code>) and paste it above</li>
                            </ol>
                            
                            <div style="margin-top: 12px; padding: 10px; background: rgba(220, 38, 38, 0.1); border-radius: 4px;">
                                <p style="margin: 0; font-size: 13px; color: #dc2626;">
                                    <strong>🚨 Security Note:</strong> Never share this secret or commit it to version control. It's used to verify webhook authenticity.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Organization Settings Tab -->
            <div id="organization" class="tab-pane">
                <h2 style="margin: 0 0 24px 0;">Organization Settings</h2>
                
                <div class="settings-grid">
                    <!-- Company Details -->
                    <div class="settings-card">
                        <h3>Company Information</h3>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Company Name</h4>
                                    <p>Company name for invoices and official documents</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['payment']['accountName'] ?? ''); ?>" 
                                           class="form-input" required>
                                    <input type="hidden" name="setting_key" value="accountName">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Company name for invoices">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Address Line 1</h4>
                                    <p>Company address first line</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['system']['companyAddress'] ?? ''); ?>" 
                                           class="form-input">
                                    <input type="hidden" name="setting_key" value="companyAddress">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Company address line 1">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Address Line 2</h4>
                                    <p>Company address second line</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['system']['companyAddress2'] ?? ''); ?>" 
                                           class="form-input">
                                    <input type="hidden" name="setting_key" value="companyAddress2">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Company address line 2">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Phone Number</h4>
                                    <p>Company phone number for invoices</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['system']['companyPhone'] ?? ''); ?>" 
                                           class="form-input">
                                    <input type="hidden" name="setting_key" value="companyPhone">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Company phone number">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Admin Email</h4>
                                    <p>Primary contact email for the system</p>
                                </div>
                                <div class="setting-control">
                                    <input type="email" name="setting_value" value="<?php echo htmlspecialchars($config['system']['adminEmail'] ?? ''); ?>" 
                                           class="form-input" required>
                                    <input type="hidden" name="setting_key" value="adminEmail">
                                    <input type="hidden" name="setting_type" value="email">
                                    <input type="hidden" name="setting_description" value="System administrator email">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Event Details -->
                    <div class="settings-card">
                        <h3>Event Information</h3>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Event Name</h4>
                                    <p>Full name of the event</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['event']['name'] ?? ''); ?>" 
                                           class="form-input" style="width: 250px;" required>
                                    <input type="hidden" name="setting_key" value="eventName">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Event name">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Event Dates</h4>
                                    <p>Date range for the event</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['event']['dates'] ?? ''); ?>" 
                                           class="form-input" style="width: 200px;" required>
                                    <input type="hidden" name="setting_key" value="eventDates">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Event dates">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Event Location</h4>
                                    <p>Venue and location details</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['event']['location'] ?? ''); ?>" 
                                           class="form-input" style="width: 250px;" required>
                                    <input type="hidden" name="setting_key" value="eventLocation">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Event location">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Payment Settings -->
                    <div class="settings-card">
                        <h3>Payment Information</h3>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Bank Name</h4>
                                    <p>Name of the bank for transfers</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['payment']['bankName'] ?? ''); ?>" 
                                           class="form-input" style="width: 250px;" required>
                                    <input type="hidden" name="setting_key" value="bankName">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Bank name for transfers">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Account Number</h4>
                                    <p>Bank account number</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['payment']['accountNumber'] ?? ''); ?>" 
                                           class="form-input" style="width: 200px;" required>
                                    <input type="hidden" name="setting_key" value="accountNumber">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Bank account number">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>SWIFT Code</h4>
                                    <p>Bank SWIFT code for international transfers</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php echo htmlspecialchars($config['payment']['swiftCode'] ?? ''); ?>" 
                                           class="form-input" style="width: 150px;" required>
                                    <input type="hidden" name="setting_key" value="swiftCode">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Bank SWIFT code">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Terms & Conditions -->
                    <div class="settings-card">
                        <h3>Terms & Conditions</h3>
                        
                        <form method="POST">
                            <div style="display: flex; flex-direction: column; gap: 15px;">
                                <div>
                                    <h4 style="margin: 0 0 8px 0;">Invoice Terms</h4>
                                    <p style="margin: 0; color: #64748b;">Terms and conditions that appear on invoices</p>
                                </div>
                                
                                <textarea name="setting_value" class="form-input" rows="10" 
                                          style="width: 100%; resize: vertical; font-family: inherit; min-height: 200px;"
                                          placeholder="Enter terms and conditions for invoices..."><?php echo htmlspecialchars($config['system']['invoiceTermsConditions'] ?? ''); ?></textarea>
                                
                                <div>
                                    <input type="hidden" name="setting_key" value="invoiceTermsConditions">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Invoice terms and conditions">
                                    <button type="submit" name="update_setting" class="btn btn-primary">Save Terms & Conditions</button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Pricing & Limits Tab -->
            <div id="pricing" class="tab-pane">
                <h2 style="margin: 0 0 24px 0;">Pricing & Registration Limits</h2>
                
                <div class="settings-grid">
                    
                    
                    
                    <!-- Dietary Options -->
                    <div class="settings-card">
                        <h3>Dietary Requirements</h3>
                        
                        <form method="POST">
                            <div class="form-group">
                                <label class="form-label">Available Options (one per line)</label>
                                <textarea name="dietary_options" class="form-input" rows="8" style="width: 100%; resize: vertical;"><?php 
                                    echo implode("\n", $config['dietaryOptions'] ?? []);
                                ?></textarea>
                            </div>
                            <button type="submit" name="update_dietary_options" class="btn btn-primary">
                                💾 Update Dietary Options
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Email Settings Tab -->
            <div id="email" class="tab-pane">
                <h2 style="margin: 0 0 24px 0;">Email Configuration</h2>
                
                <div class="settings-grid">
                    <!-- Brevo API Settings -->
                    <div class="settings-card">
                        <h3>🔑 Brevo API Configuration</h3>
                        <p>Configure your Brevo email service settings for sending registration and notification emails.</p>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Brevo API Key</h4>
                                    <p>Your Brevo API key for sending emails</p>
                                    <small style="color: #64748b;">Get your API key from <a href="https://app.brevo.com/settings/keys/api" target="_blank">Brevo Dashboard</a></small>
                                </div>
                                <div class="setting-control">
                                    <input type="password" name="setting_value" value="<?php 
                                        $apiKey = '';
                                        if (isset($db_settings)) {
                                            foreach ($db_settings as $setting) {
                                                if ($setting['setting_key'] === 'brevoApiKey') {
                                                    $apiKey = $setting['setting_value'];
                                                    break;
                                                }
                                            }
                                        }
                                        echo htmlspecialchars($apiKey);
                                    ?>" class="form-input" style="width: 300px;" placeholder="xkeysib-...">
                                    <input type="hidden" name="setting_key" value="brevoApiKey">
                                    <input type="hidden" name="setting_type" value="string">
                                    <input type="hidden" name="setting_description" value="Brevo API Key">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Sender Settings -->
                    <div class="settings-card">
                        <h3>📧 Email Sender Configuration</h3>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Sender Email Address</h4>
                                    <p>Email address used as sender for all outgoing emails</p>
                                </div>
                                <div class="setting-control">
                                    <input type="email" name="setting_value" value="<?php 
                                        $senderEmail = '';
                                        if (isset($db_settings)) {
                                            foreach ($db_settings as $setting) {
                                                if ($setting['setting_key'] === 'brevoSenderEmail') {
                                                    $senderEmail = $setting['setting_value'];
                                                    break;
                                                }
                                            }
                                        }
                                        echo htmlspecialchars($senderEmail);
                                    ?>" class="form-input" style="width: 250px;" required>
                                    <input type="hidden" name="setting_key" value="brevoSenderEmail">
                                    <input type="hidden" name="setting_type" value="email">
                                    <input type="hidden" name="setting_description" value="Brevo sender email address">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Sender Name</h4>
                                    <p>Display name for the email sender</p>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php 
                                        $senderName = '';
                                        if (isset($db_settings)) {
                                            foreach ($db_settings as $setting) {
                                                if ($setting['setting_key'] === 'brevoSenderName') {
                                                    $senderName = $setting['setting_value'];
                                                    break;
                                                }
                                            }
                                        }
                                        echo htmlspecialchars($senderName);
                                    ?>" class="form-input" style="width: 200px;" required>
                                    <input type="hidden" name="setting_key" value="brevoSenderName">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Brevo sender name">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Admin Notifications -->
                    <div class="settings-card">
                        <h3>🔔 Admin Notifications</h3>
                        
                        <form method="POST">
                            <div class="setting-row">
                                <div class="setting-info">
                                    <h4>Admin Notification Emails</h4>
                                    <p>Email addresses to receive admin notifications (comma-separated)</p>
                                    <small style="color: #64748b;">Multiple emails: admin@example.com, manager@example.com</small>
                                </div>
                                <div class="setting-control">
                                    <input type="text" name="setting_value" value="<?php 
                                        $adminEmails = '';
                                        if (isset($db_settings)) {
                                            foreach ($db_settings as $setting) {
                                                if ($setting['setting_key'] === 'adminNotificationEmails') {
                                                    $adminEmails = $setting['setting_value'];
                                                    break;
                                                }
                                            }
                                        }
                                        echo htmlspecialchars($adminEmails);
                                    ?>" class="form-input" style="width: 350px;" required>
                                    <input type="hidden" name="setting_key" value="adminNotificationEmails">
                                    <input type="hidden" name="setting_type" value="text">
                                    <input type="hidden" name="setting_description" value="Admin notification email addresses (comma-separated)">
                                    <button type="submit" name="update_setting" class="btn btn-sm">Save</button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Bank Transfer Instructions -->
                    <div class="settings-card">
                        <h3>🏦 Bank Transfer Instructions</h3>
                        <p>Instructions included in pending registration emails for bank transfer payments.</p>
                        
                        <form method="POST">
                            <div style="margin-bottom: 20px;">
                                <h4 style="margin: 0 0 8px 0;">Bank Transfer Details</h4>
                                <p style="margin: 0 0 12px 0; color: #64748b;">These details will be included in emails sent to users who choose bank transfer payment.</p>
                                <textarea name="setting_value" class="form-input" rows="8" style="width: 100%; min-height: 200px; resize: vertical; box-sizing: border-box;"><?php 
                                    $bankDetails = '';
                                    if (isset($db_settings)) {
                                        foreach ($db_settings as $setting) {
                                            if ($setting['setting_key'] === 'bankTransferDetails') {
                                                $bankDetails = $setting['setting_value'];
                                                break;
                                            }
                                        }
                                    }
                                    if (empty($bankDetails)) {
                                        $bankDetails = "Bank Name: DBS Bank Ltd Singapore\nAccount Name: AppliedHE Pte. Ltd.\nAccount Number: 1234567890\nSWIFT Code: DBSSSGSG\n\nPlease include your registration number as the payment reference.";
                                    }
                                    echo htmlspecialchars($bankDetails);
                                ?></textarea>
                                <input type="hidden" name="setting_key" value="bankTransferDetails">
                                <input type="hidden" name="setting_type" value="text">
                                <input type="hidden" name="setting_description" value="Bank transfer instructions for emails">
                                <button type="submit" name="update_setting" class="btn" style="margin-top: 12px;">Update Bank Details</button>
                            </div>
                        </form>
                        
                        <div style="margin-top: 30px; padding: 20px; background: #f8fafc; border-radius: 8px; border-left: 4px solid #3b82f6;">
                            <h4 style="margin: 0 0 12px 0; color: #1e40af;">Preview in Email</h4>
                            <div style="font-size: 14px; line-height: 1.6; color: #475569;">
                                <?php 
                                $previewBank = !empty($bankDetails) ? $bankDetails : "Bank Name: DBS Bank Ltd Singapore\nAccount Name: AppliedHE Pte. Ltd.\nAccount Number: 1234567890\nSWIFT Code: DBSSSGSG\n\nPlease include your registration number as the payment reference.";
                                echo nl2br(htmlspecialchars($previewBank)); 
                                ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Email Testing -->
                    <div class="settings-card">
                        <h3>🧪 Email Testing & Validation</h3>
                        <p>Test your email configuration and verify API connectivity.</p>
                        
                        <!-- API Connection Test -->
                        <div style="margin: 20px 0; padding: 20px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0;">
                            <h4 style="margin: 0 0 12px 0;">🔗 API Connection Test</h4>
                            <p style="margin: 0 0 15px 0; color: #64748b;">Verify that your Brevo API key is valid and working.</p>
                            
                            <button id="testApiBtn" class="btn btn-primary" style="margin-right: 12px;">
                                🔍 Test API Connection
                            </button>
                            
                            <div id="apiTestResult" style="margin-top: 15px; display: none;"></div>
                        </div>
                        
                        <!-- Send Test Email -->
                        <div style="margin: 20px 0; padding: 20px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0;">
                            <h4 style="margin: 0 0 12px 0;">📧 Send Test Email</h4>
                            <p style="margin: 0 0 15px 0; color: #64748b;">Send a test email to verify end-to-end email delivery.</p>
                            
                            <div style="display: flex; gap: 12px; align-items: center; margin-bottom: 15px;">
                                <input type="email" id="testEmailInput" placeholder="Enter test email address" 
                                       class="form-input" style="width: 250px;" required>
                                <button id="sendTestBtn" class="btn btn-primary">
                                    📤 Send Test Email
                                </button>
                            </div>
                            
                            <div id="emailTestResult" style="margin-top: 15px; display: none;"></div>
                        </div>
                        
                        <!-- Quick Links -->
                        <div style="margin: 20px 0;">
                            <h4 style="margin: 0 0 12px 0;">📋 Email Management</h4>
                            <div style="display: flex; gap: 12px;">
                                <a href="email_templates.php" class="btn btn-secondary">
                                    📧 View Email Templates
                                </a>
                                <a href="email_logs.php" class="btn btn-secondary">
                                    📊 View Email Logs
                                </a>
                            </div>
                        </div>
                        
                        <div style="margin-top: 20px; padding: 15px; background: #fef3c7; border-radius: 8px; border-left: 4px solid #f59e0b;">
                            <h4 style="margin: 0 0 8px 0; color: #92400e;">⚠️ Important Notes</h4>
                            <ul style="margin: 0; padding-left: 20px; color: #78350f;">
                                <li>Make sure your Brevo API key is valid and active</li>
                                <li>Verify sender email is authorized in your Brevo account</li>
                                <li>Test email templates before going live</li>
                                <li>Monitor email logs for delivery issues</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- System Management Tab -->
            <div id="system" class="tab-pane">
                <h2 style="margin: 0 0 24px 0;">System Management</h2>
                
                <div class="settings-grid">
                    <!-- Database Management -->
                    <div class="settings-card">
                        <h3>Database Settings Management</h3>
                        
                        <div style="margin-bottom: 20px;">
                            <p class="muted">
                                System now uses database-only configuration. All settings are stored in the admin_settings table.
                            </p>
                        </div>
                        
                        <form method="POST">
                            <button type="submit" name="reset_to_defaults" class="btn btn-warning" 
                                    onclick="return confirm('This will reset ALL settings to default values. This action cannot be undone. Continue?')">
                                🔄 Reset to Default Settings
                            </button>
                        </form>
                        
                        <div style="margin-top: 16px;">
                            <a href="backup-config.php" class="btn btn-secondary">
                                💾 Download Config Backup
                            </a>
                        </div>
                        
                        <div style="margin-top: 12px; padding: 12px; background: var(--slate-50); border-radius: 6px;">
                            <p style="margin: 0; font-size: 14px; color: var(--slate-600);">
                                <strong>Status:</strong> Database-only configuration active<br>
                                <strong>Settings Count:</strong> <?php echo number_format($db_settings_count); ?> entries
                            </p>
                        </div>
                    </div>
                    
                    <!-- Current Settings Overview -->
                    <div class="settings-card">
                        <h3>Settings Overview</h3>
                        
                        <div style="max-height: 300px; overflow-y: auto;">
                            <?php if (!empty($db_settings)): ?>
                                <?php foreach ($db_settings as $setting): ?>
                                <div style="padding: 8px 0; border-bottom: 1px solid var(--slate-200); font-size: 14px;">
                                    <strong><?php echo htmlspecialchars($setting['setting_key']); ?>:</strong>
                                    <span style="color: var(--slate-600);">
                                        <?php echo htmlspecialchars($setting['setting_value']); ?>
                                    </span>
                                </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <p class="muted">No database settings found. Using JSON file configuration.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        function showTab(tabName, updateUrl = true) {
            // Hide all tab panes
            document.querySelectorAll('.tab-pane').forEach(pane => {
                pane.classList.remove('active');
            });
            
            // Remove active class from all tab buttons
            document.querySelectorAll('.tab-button').forEach(btn => {
                btn.classList.remove('active');
            });
            
            // Show selected tab pane
            const targetPane = document.getElementById(tabName);
            if (targetPane) {
                targetPane.classList.add('active');
            }
            
            // Add active class to corresponding button
            const targetButton = document.querySelector(`[onclick="showTab('${tabName}')"]`);
            if (targetButton) {
                targetButton.classList.add('active');
            }
            
            // Update URL hash if requested
            if (updateUrl) {
                window.location.hash = tabName;
            }
        }
        
        // Email testing functions
        async function testApiConnection() {
            const btn = document.getElementById('testApiBtn');
            const resultDiv = document.getElementById('apiTestResult');
            
            btn.disabled = true;
            btn.innerHTML = '🔄 Testing...';
            resultDiv.style.display = 'none';
            
            try {
                const response = await fetch('test_email_api.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        test_type: 'api_connection'
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    resultDiv.innerHTML = `
                        <div style="padding: 15px; background: #d1fae5; border: 1px solid #10b981; border-radius: 5px; color: #065f46;">
                            <h4 style="margin: 0 0 10px 0;">✅ API Connection Successful!</h4>
                            <p style="margin: 0 0 10px 0;">${data.message}</p>
                            ${data.account_info ? `
                                <div style="margin-top: 10px;">
                                    <strong>Account Details:</strong><br>
                                    <small>
                                        Email: ${data.account_info.email}<br>
                                        Name: ${data.account_info.firstName} ${data.account_info.lastName}<br>
                                        Company: ${data.account_info.companyName}<br>
                                        Plan: ${data.account_info.plan}
                                    </small>
                                </div>
                            ` : ''}
                        </div>
                    `;
                } else {
                    resultDiv.innerHTML = `
                        <div style="padding: 15px; background: #fee2e2; border: 1px solid #dc2626; border-radius: 5px; color: #991b1b;">
                            <h4 style="margin: 0 0 10px 0;">❌ API Connection Failed</h4>
                            <p style="margin: 0;">${data.error}</p>
                        </div>
                    `;
                }
                
            } catch (error) {
                resultDiv.innerHTML = `
                    <div style="padding: 15px; background: #fee2e2; border: 1px solid #dc2626; border-radius: 5px; color: #991b1b;">
                        <h4 style="margin: 0 0 10px 0;">❌ Connection Error</h4>
                        <p style="margin: 0;">Failed to test API connection: ${error.message}</p>
                    </div>
                `;
            } finally {
                btn.disabled = false;
                btn.innerHTML = '🔍 Test API Connection';
                resultDiv.style.display = 'block';
            }
        }
        
        async function sendTestEmail() {
            const btn = document.getElementById('sendTestBtn');
            const emailInput = document.getElementById('testEmailInput');
            const resultDiv = document.getElementById('emailTestResult');
            
            const testEmail = emailInput.value.trim();
            if (!testEmail) {
                alert('Please enter a test email address');
                return;
            }
            
            btn.disabled = true;
            btn.innerHTML = '📤 Sending...';
            resultDiv.style.display = 'none';
            
            try {
                const response = await fetch('test_email_api.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        test_type: 'send_test_email',
                        test_email: testEmail
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    resultDiv.innerHTML = `
                        <div style="padding: 15px; background: #d1fae5; border: 1px solid #10b981; border-radius: 5px; color: #065f46;">
                            <h4 style="margin: 0 0 10px 0;">✅ Test Email Sent Successfully!</h4>
                            <p style="margin: 0 0 10px 0;">${data.message}</p>
                            <div style="margin-top: 10px;">
                                <small>
                                    <strong>Recipient:</strong> ${data.recipient}<br>
                                    ${data.message_id ? `<strong>Message ID:</strong> ${data.message_id}<br>` : ''}
                                    <strong>Time:</strong> ${new Date().toLocaleString()}
                                </small>
                            </div>
                            <p style="margin: 10px 0 0 0; font-size: 14px;">
                                📬 Check your inbox (and spam folder) for the test email.
                            </p>
                        </div>
                    `;
                } else {
                    resultDiv.innerHTML = `
                        <div style="padding: 15px; background: #fee2e2; border: 1px solid #dc2626; border-radius: 5px; color: #991b1b;">
                            <h4 style="margin: 0 0 10px 0;">❌ Failed to Send Test Email</h4>
                            <p style="margin: 0;">${data.error}</p>
                        </div>
                    `;
                }
                
            } catch (error) {
                resultDiv.innerHTML = `
                    <div style="padding: 15px; background: #fee2e2; border: 1px solid #dc2626; border-radius: 5px; color: #991b1b;">
                        <h4 style="margin: 0 0 10px 0;">❌ Connection Error</h4>
                        <p style="margin: 0;">Failed to send test email: ${error.message}</p>
                    </div>
                `;
            } finally {
                btn.disabled = false;
                btn.innerHTML = '📤 Send Test Email';
                resultDiv.style.display = 'block';
            }
        }
        
        // Handle browser back/forward navigation
        window.addEventListener('hashchange', function() {
            const hash = window.location.hash.substring(1);
            if (hash && document.getElementById(hash)) {
                showTab(hash, false);
            }
        });
        
        // Initialize tab based on URL hash on page load
        document.addEventListener('DOMContentLoaded', function() {
            const hash = window.location.hash.substring(1);
            const validTabs = ['general', 'organization', 'pricing', 'email', 'system'];
            
            if (hash && validTabs.includes(hash)) {
                showTab(hash, false);
            } else {
                // Default to general tab if no valid hash
                showTab('general', false);
            }
            
            // Add event listeners for email testing buttons
            const testApiBtn = document.getElementById('testApiBtn');
            const sendTestBtn = document.getElementById('sendTestBtn');
            const testEmailInput = document.getElementById('testEmailInput');
            
            if (testApiBtn) {
                testApiBtn.addEventListener('click', testApiConnection);
            }
            
            if (sendTestBtn) {
                sendTestBtn.addEventListener('click', sendTestEmail);
            }
            
            if (testEmailInput) {
                testEmailInput.addEventListener('keypress', function(e) {
                    if (e.key === 'Enter') {
                        sendTestEmail();
                    }
                });
            }
        });
        
        // Toggle webhook secret visibility
        function toggleWebhookSecret() {
            const displaySpan = document.getElementById('webhook-secret-display');
            const fullSpan = document.getElementById('webhook-secret-full');
            const toggleBtn = document.getElementById('toggle-secret');
            
            if (displaySpan && fullSpan && toggleBtn) {
                if (displaySpan.style.display === 'none') {
                    // Currently showing full, switch to masked
                    displaySpan.style.display = 'inline';
                    fullSpan.style.display = 'none';
                    toggleBtn.innerHTML = '👁️ Show';
                } else {
                    // Currently showing masked, switch to full
                    displaySpan.style.display = 'none';
                    fullSpan.style.display = 'inline';
                    toggleBtn.innerHTML = '🙈 Hide';
                }
            }
        }
    </script>

<?php include 'includes/footer.php'; ?>
