<?php
/**
 * Clean URLs Verification Script
 * This script checks if mod_rewrite and .htaccess are working correctly
 */

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Clean URLs Verification</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 40px 20px;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }
        
        .header p {
            font-size: 16px;
            opacity: 0.9;
        }
        
        .content {
            padding: 40px;
        }
        
        .check-item {
            display: flex;
            align-items: center;
            padding: 20px;
            margin-bottom: 15px;
            border-radius: 8px;
            background: #f8f9fa;
            border-left: 4px solid #6c757d;
        }
        
        .check-item.success {
            background: #d4edda;
            border-left-color: #28a745;
        }
        
        .check-item.error {
            background: #f8d7da;
            border-left-color: #dc3545;
        }
        
        .check-item.warning {
            background: #fff3cd;
            border-left-color: #ffc107;
        }
        
        .check-icon {
            font-size: 24px;
            margin-right: 15px;
            min-width: 30px;
            text-align: center;
        }
        
        .check-content {
            flex: 1;
        }
        
        .check-title {
            font-weight: 600;
            margin-bottom: 5px;
            font-size: 16px;
        }
        
        .check-description {
            font-size: 14px;
            color: #6c757d;
            line-height: 1.5;
        }
        
        .test-links {
            margin-top: 30px;
            padding: 25px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        
        .test-links h3 {
            margin-bottom: 15px;
            color: #333;
        }
        
        .test-link {
            display: block;
            padding: 12px 15px;
            margin-bottom: 10px;
            background: white;
            border: 2px solid #dee2e6;
            border-radius: 6px;
            text-decoration: none;
            color: #667eea;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        
        .test-link:hover {
            background: #667eea;
            color: white;
            transform: translateX(5px);
        }
        
        .instructions {
            margin-top: 30px;
            padding: 25px;
            background: #e7f3ff;
            border-radius: 8px;
            border-left: 4px solid #0066cc;
        }
        
        .instructions h3 {
            color: #0066cc;
            margin-bottom: 15px;
        }
        
        .instructions ol {
            margin-left: 20px;
        }
        
        .instructions li {
            margin-bottom: 10px;
            line-height: 1.6;
        }
        
        .code-block {
            background: #2d3748;
            color: #e2e8f0;
            padding: 15px;
            border-radius: 6px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            margin: 10px 0;
            overflow-x: auto;
        }
        
        .footer {
            padding: 20px 40px;
            background: #f8f9fa;
            text-align: center;
            color: #6c757d;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚀 Clean URLs Verification</h1>
            <p>Check if your URL rewriting is configured correctly</p>
        </div>
        
        <div class="content">
            <?php
            $checks = [];
            
            // Check 1: .htaccess file exists
            $htaccess_exists = file_exists(__DIR__ . '/.htaccess');
            $checks[] = [
                'status' => $htaccess_exists ? 'success' : 'error',
                'icon' => $htaccess_exists ? '✅' : '❌',
                'title' => '.htaccess File',
                'description' => $htaccess_exists 
                    ? 'The .htaccess file exists in the admin directory.' 
                    : 'The .htaccess file is missing! Clean URLs will not work.'
            ];
            
            // Check 2: mod_rewrite detection
            $mod_rewrite_enabled = false;
            if (function_exists('apache_get_modules')) {
                $modules = apache_get_modules();
                $mod_rewrite_enabled = in_array('mod_rewrite', $modules);
            } else {
                // Alternative check via .htaccess
                $mod_rewrite_enabled = getenv('REDIRECT_STATUS') !== false || isset($_SERVER['REDIRECT_STATUS']);
            }
            
            $checks[] = [
                'status' => $mod_rewrite_enabled ? 'success' : 'warning',
                'icon' => $mod_rewrite_enabled ? '✅' : '⚠️',
                'title' => 'mod_rewrite Module',
                'description' => $mod_rewrite_enabled 
                    ? 'The mod_rewrite module appears to be enabled.' 
                    : 'Cannot confirm if mod_rewrite is enabled. Please verify manually in Apache configuration.'
            ];
            
            // Check 3: Current URL structure
            $current_url = $_SERVER['REQUEST_URI'];
            $has_php_extension = strpos($current_url, '.php') !== false;
            
            $checks[] = [
                'status' => $has_php_extension ? 'warning' : 'success',
                'icon' => $has_php_extension ? '⚠️' : '✅',
                'title' => 'URL Structure',
                'description' => $has_php_extension 
                    ? 'You are accessing this page with .php extension. Try accessing without it for clean URLs.' 
                    : 'You are using clean URLs! The URL rewriting is working.'
            ];
            
            // Check 4: Write permissions
            $writable = is_writable(__DIR__);
            $checks[] = [
                'status' => $writable ? 'success' : 'warning',
                'icon' => $writable ? '✅' : '⚠️',
                'title' => 'Directory Permissions',
                'description' => $writable 
                    ? 'The admin directory has proper write permissions.' 
                    : 'Directory may have restricted permissions. This could affect .htaccess functionality.'
            ];
            
            // Display all checks
            foreach ($checks as $check) {
                echo '<div class="check-item ' . $check['status'] . '">';
                echo '<div class="check-icon">' . $check['icon'] . '</div>';
                echo '<div class="check-content">';
                echo '<div class="check-title">' . $check['title'] . '</div>';
                echo '<div class="check-description">' . $check['description'] . '</div>';
                echo '</div>';
                echo '</div>';
            }
            ?>
            
            <div class="test-links">
                <h3>🧪 Test These URLs</h3>
                <p style="margin-bottom: 15px; color: #6c757d;">Try accessing these pages with clean URLs:</p>
                
                <a href="dashboard" class="test-link" target="_blank">
                    📊 Dashboard (Clean URL)
                </a>
                
                <a href="registrations" class="test-link" target="_blank">
                    👥 Registrations (Clean URL)
                </a>
                
                <a href="delegates" class="test-link" target="_blank">
                    🎓 Delegates (Clean URL)
                </a>
                
                <a href="settings" class="test-link" target="_blank">
                    ⚙️ Settings (Clean URL)
                </a>
            </div>
            
            <?php if (!$mod_rewrite_enabled || !$htaccess_exists): ?>
            <div class="instructions">
                <h3>📋 Setup Instructions</h3>
                <p><strong>Follow these steps to enable clean URLs:</strong></p>
                
                <ol>
                    <li>
                        <strong>Enable mod_rewrite in Apache:</strong>
                        <ul style="margin-top: 5px;">
                            <li>Open XAMPP Control Panel</li>
                            <li>Click "Config" next to Apache → Select "httpd.conf"</li>
                            <li>Find: <code style="background: #f8f9fa; padding: 2px 6px; border-radius: 3px;">#LoadModule rewrite_module modules/mod_rewrite.so</code></li>
                            <li>Remove the <code style="background: #f8f9fa; padding: 2px 6px; border-radius: 3px;">#</code> to uncomment it</li>
                        </ul>
                    </li>
                    
                    <li>
                        <strong>Allow .htaccess overrides:</strong>
                        <ul style="margin-top: 5px;">
                            <li>In the same httpd.conf file, find <code style="background: #f8f9fa; padding: 2px 6px; border-radius: 3px;">&lt;Directory "C:/xampp/htdocs"&gt;</code></li>
                            <li>Change <code style="background: #f8f9fa; padding: 2px 6px; border-radius: 3px;">AllowOverride None</code> to <code style="background: #f8f9fa; padding: 2px 6px; border-radius: 3px;">AllowOverride All</code></li>
                        </ul>
                    </li>
                    
                    <li>
                        <strong>Restart Apache</strong> in XAMPP Control Panel
                    </li>
                    
                    <li>
                        <strong>Refresh this page</strong> to verify the setup
                    </li>
                </ol>
                
                <p style="margin-top: 15px;">
                    <strong>📖 Need detailed instructions?</strong> 
                    Check the <code style="background: #f8f9fa; padding: 2px 6px; border-radius: 3px;">CLEAN_URLS_SETUP.md</code> file in the project root.
                </p>
            </div>
            <?php endif; ?>
            
            <div style="margin-top: 30px; padding: 20px; background: #f8f9fa; border-radius: 8px; text-align: center;">
                <p style="color: #6c757d; margin-bottom: 10px;">
                    Current URL: <code style="background: white; padding: 5px 10px; border-radius: 4px; color: #667eea;"><?php echo htmlspecialchars($_SERVER['REQUEST_URI']); ?></code>
                </p>
                <p style="color: #6c757d; font-size: 14px;">
                    Server: <?php echo $_SERVER['SERVER_SOFTWARE']; ?>
                </p>
            </div>
        </div>
        
        <div class="footer">
            <p>🎯 Once everything shows ✅, your clean URLs are ready to use!</p>
        </div>
    </div>
</body>
</html>

