# Delegate Email Field - Data Flow Diagram

## 📊 System Architecture Overview

```
┌─────────────────────────────────────────────────────────────────────┐
│                         USER REGISTRATION FLOW                       │
└─────────────────────────────────────────────────────────────────────┘

FRONTEND                     API LAYER              DATABASE
┌──────────┐               ┌──────────┐            ┌──────────┐
│          │               │          │            │          │
│  Step 1  │               │          │            │  MySQL   │
│ Select   │               │          │            │          │
│ Products │               │          │            │          │
│          │               │          │            │          │
└────┬─────┘               │          │            │          │
     │                     │          │            │          │
     ▼                     │          │            │          │
┌──────────┐               │          │            │          │
│  Step 2  │◄──────────────┼──────────┼────────────┤          │
│ Delegate │  Dynamic      │          │            │          │
│ Details  │  Form         │          │            │          │
│          │  Generation   │          │            │          │
│  ┌─────┐ │               │          │            │          │
│  │Email│ │◄── NEW FIELD  │          │            │          │
│  └─────┘ │               │          │            │          │
└────┬─────┘               │          │            │          │
     │                     │          │            │          │
     │ formData.delegates  │          │            │          │
     │ [{fullName, badge,  │          │            │          │
     │   email, ... }]     │          │            │          │
     ▼                     │          │            │          │
┌──────────┐               │          │            │          │
│  Step 3  │               │          │            │          │
│ Contact  │               │          │            │          │
│   Info   │               │          │            │          │
└────┬─────┘               │          │            │          │
     │                     │          │            │          │
     ▼                     │          │            │          │
┌──────────┐               │          │            │          │
│  Step 4  │               │          │            │          │
│  Review  │               │          │            │          │
│          │               │          │            │          │
│  Shows:  │               │          │            │          │
│  • Name  │               │          │            │          │
│  • Email │◄── DISPLAYED  │          │            │          │
│  • Badge │               │          │            │          │
└────┬─────┘               │          │            │          │
     │                     │          │            │          │
     │ Submit Form         │          │            │          │
     ▼                     ▼          │            │          │
┌──────────┐         ┌──────────┐    │            │          │
│  Step 5  │────────►│ api.php  │    │            │          │
│ Payment  │  POST   │          │    │            │          │
└──────────┘  JSON   │ register │    │            │          │
              Data   │ action   │    │            │          │
                     │          │────┼───────────►│delegates │
                     │ INSERT   │    │            │  table   │
                     │ delegates│    │            │          │
                     │ with     │    │            │  id      │
                     │ email    │    │            │  ...     │
                     │          │    │            │  email   │◄─NEW
                     └──────────┘    │            │  ...     │
                                     │            │          │
                                     │            └──────────┘
                                     │
                                     │
┌─────────────────────────────────────────────────────────────────────┐
│                         ADMIN BACKEND FLOW                           │
└─────────────────────────────────────────────────────────────────────┘

ADMIN INTERFACE              API/BACKEND             DATABASE
┌──────────────┐            ┌──────────┐           ┌──────────┐
│              │            │          │           │          │
│  Dashboard   │◄───────────┤  Stats   │◄──────────┤delegates │
│              │  Display   │  Queries │   SELECT  │          │
│  • Total     │            │          │           │  email   │
│    Delegates │            │          │           │          │
└──────────────┘            └──────────┘           └──────────┘
                                   ▲                     │
                                   │                     │
┌──────────────┐                   │                     │
│              │                   │                     │
│  Delegates   │◄──────────────────┼─────────────────────┘
│  List Page   │    Display        │
│              │                   │
│  Table with: │                   │
│  • Name      │                   │
│  • Badge     │                   │
│  • EMAIL ◄───┼───────────────────┘ NEW COLUMN
│  • Univ.     │
│              │
│  [Edit] btn  │─────┐
└──────────────┘     │
                     │
                     ▼
              ┌──────────────┐
              │ Edit Modal   │
              │              │
              │  ┌─────┐     │
              │  │Email│◄────┼── NEW FIELD
              │  └─────┘     │
              │              │
              │  [Save]      │
              └──────┬───────┘
                     │
                     │ POST update
                     ▼
              ┌──────────────┐         ┌──────────┐
              │ delegates.php│────────►│ UPDATE   │
              │ UPDATE       │  SQL    │delegates │
              │ handler      │         │SET email │
              └──────────────┘         └──────────┘

┌──────────────┐            ┌──────────┐           ┌──────────┐
│              │            │          │           │          │
│Registration  │◄───────────┤ SELECT   │◄──────────┤delegates │
│Details Page  │  Display   │ delegates│   JOIN    │  email   │
│              │            │          │           │          │
│  Delegate    │            │          │           │          │
│  Table with  │            │          │           │          │
│  EMAIL       │◄───────────┘          │           │          │
└──────────────┘                       │           └──────────┘
                                       │
┌──────────────┐                       │
│              │                       │
│  Export CSV  │◄──────────────────────┘
│              │       SELECT delegates
│  Includes    │       with email column
│  email       │
│  column      │
└──────────────┘


┌─────────────────────────────────────────────────────────────────────┐
│                       EMAIL NOTIFICATIONS (FUTURE)                   │
└─────────────────────────────────────────────────────────────────────┘

                     ┌──────────────┐
                     │email_service │
                     │    .php      │
                     │              │
                     │  Optional:   │
                     │  Send to     │
                     │  individual  │
                     │  delegates   │
                     └──────┬───────┘
                            │
                            │ Query delegates
                            │ with email
                            ▼
                     ┌──────────────┐
                     │   SELECT     │
                     │   email      │
                     │   FROM       │
                     │   delegates  │
                     └──────┬───────┘
                            │
                            │ Send emails
                            ▼
                     ┌──────────────┐
                     │ Individual   │
                     │ Delegate     │
                     │ Emails       │
                     └──────────────┘
```

---

## 🔄 Data Transformation Flow

### 1. Data Collection (Frontend)

```javascript
// registration.js - Step 2
Input Field:
  <input type="email" id="delegate1Email" name="delegate1Email" />

Collected as:
  formData.delegates = [
    {
      fullName: "John Doe",
      badgeName: "John",
      designation: "Professor",
      email: "john.doe@university.edu", // ← NEW
      dietary: "Vegetarian",
      assistance: ""
    }
  ]
```

### 2. Data Transmission (API)

```javascript
// POST to api.php
{
  "delegates": [
    {
      "fullName": "John Doe",
      "badgeName": "John", 
      "designation": "Professor",
      "email": "john.doe@university.edu", // ← NEW
      "dietary": "Vegetarian",
      "assistance": ""
    }
  ],
  // ... other registration data
}
```

### 3. Data Storage (Database)

```sql
-- api.php executes:
INSERT INTO delegates (
  registration_id, delegate_number, 
  full_name, badge_name, designation, 
  email,  -- ← NEW COLUMN
  dietary_requirements, special_assistance
) VALUES (
  1, 1, 
  'John Doe', 'John', 'Professor',
  'john.doe@university.edu',  -- ← NEW VALUE
  'Vegetarian', ''
);
```

### 4. Data Retrieval (Admin Panel)

```sql
-- admin/delegates.php
SELECT 
  d.id,
  d.full_name,
  d.badge_name,
  d.email,  -- ← NEW FIELD
  d.designation,
  r.university_name,
  r.payment_status
FROM delegates d
JOIN registrations r ON d.registration_id = r.id;
```

### 5. Data Display (HTML)

```php
<!-- admin/delegates.php -->
<table>
  <tr>
    <td>John Doe</td>
    <td>John</td>
    <td>john.doe@university.edu</td>  <!-- ← NEW COLUMN -->
    <td>Professor</td>
  </tr>
</table>
```

---

## 🗂️ Database Schema Changes

### BEFORE (Current)
```sql
CREATE TABLE delegates (
  id INT AUTO_INCREMENT PRIMARY KEY,
  registration_id INT NOT NULL,
  delegate_number INT NOT NULL,
  full_name VARCHAR(255) NOT NULL,
  badge_name VARCHAR(100) NOT NULL,
  designation VARCHAR(255) NOT NULL,
  dietary_requirements VARCHAR(100),
  special_assistance TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  
  FOREIGN KEY (registration_id) REFERENCES registrations(id),
  INDEX idx_registration_id (registration_id)
);
```

### AFTER (With Email)
```sql
CREATE TABLE delegates (
  id INT AUTO_INCREMENT PRIMARY KEY,
  registration_id INT NOT NULL,
  delegate_number INT NOT NULL,
  full_name VARCHAR(255) NOT NULL,
  badge_name VARCHAR(100) NOT NULL,
  designation VARCHAR(255) NOT NULL,
  email VARCHAR(255) DEFAULT NULL,  -- ← NEW FIELD
  dietary_requirements VARCHAR(100),
  special_assistance TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  
  FOREIGN KEY (registration_id) REFERENCES registrations(id),
  INDEX idx_registration_id (registration_id),
  INDEX idx_delegate_email (email)  -- ← NEW INDEX
);
```

---

## 📁 File Dependencies Map

```
delegates table (email column)
    │
    ├──► FRONTEND
    │     └─ registration.js
    │         ├─ createDelegateCard()      [Display field]
    │         ├─ saveCurrentStepData()     [Collect value]
    │         ├─ populateReview()          [Show in review]
    │         └─ restoreFormFields()       [Restore on reload]
    │
    ├──► API
    │     └─ api.php
    │         └─ handleRegistration()      [Insert to DB]
    │
    ├──► ADMIN PAGES
    │     ├─ delegates.php
    │     │   ├─ UPDATE handler            [Edit delegate]
    │     │   ├─ Table display             [Show in list]
    │     │   └─ Edit modal                [Edit form]
    │     │
    │     ├─ registration-details.php
    │     │   └─ Delegate table            [Show in details]
    │     │
    │     └─ export.php
    │         └─ CSV export query          [Include in export]
    │
    └──► OPTIONAL
          └─ email_service.php              [Future use]
```

---

## 🔀 Data States

### State 1: Empty (NULL)
```
Database: email = NULL
Display:  "—" or empty
Meaning:  No email provided
```

### State 2: Valid Email
```
Database: email = "john@university.edu"
Display:  "john@university.edu"
Meaning:  Valid email provided
```

### State 3: Invalid Email (Should be prevented)
```
Form Validation: Prevents submission
Browser: Shows "Please enter a valid email"
```

---

## 🎯 Touch Points Summary

| Layer | Component | Action | Line Ref |
|-------|-----------|--------|----------|
| **Database** | delegates table | Add column | Migration SQL |
| **Frontend** | registration.js | Display field | ~1614 |
| **Frontend** | registration.js | Save value | ~1755 |
| **Frontend** | registration.js | Show in review | ~1912 |
| **Frontend** | registration.js | Restore value | ~2565 |
| **Backend** | api.php | Insert to DB | ~304-320 |
| **Admin** | delegates.php | Update handler | ~29-42 |
| **Admin** | delegates.php | Table header | ~373 |
| **Admin** | delegates.php | Table row | ~400 |
| **Admin** | delegates.php | Edit modal | ~520 |
| **Admin** | delegates.php | Modal function | ~691 |
| **Admin** | registration-details.php | Table header | ~1510 |
| **Admin** | registration-details.php | Table row | ~1529 |
| **Admin** | export.php | CSV query | ~52 |

**Total Touch Points: 14**

---

## 🛡️ Data Validation Flow

```
┌─────────────┐
│ User Input  │
│ (email)     │
└──────┬──────┘
       │
       ▼
┌─────────────────┐
│ HTML5 Browser   │◄── type="email" attribute
│ Validation      │    Checks format: x@y.z
└──────┬──────────┘
       │ Valid
       ▼
┌─────────────────┐
│ JavaScript      │◄── Optional: Additional checks
│ (Optional)      │
└──────┬──────────┘
       │
       ▼
┌─────────────────┐
│ API Receives    │
│ Data            │
└──────┬──────────┘
       │
       ▼
┌─────────────────┐
│ PHP Sanitize    │◄── $delegate['email'] ?? null
│ (api.php)       │    Handles missing values
└──────┬──────────┘
       │
       ▼
┌─────────────────┐
│ Database Insert │◄── VARCHAR(255) NULL
│                 │    Accepts NULL or string
└─────────────────┘
       │
       ▼
┌─────────────────┐
│ Display         │◄── htmlspecialchars($email)
│ (XSS Protected) │    Prevents code injection
└─────────────────┘
```

---

## 📊 Impact Matrix

| Area | Files | Complexity | Risk | Time |
|------|-------|------------|------|------|
| Database | 4 | Low | Low | 30m |
| Frontend | 1 | Medium | Low | 2h |
| Backend API | 1 | Low | Low | 1h |
| Admin Panel | 3 | Medium | Low | 2.5h |
| Testing | N/A | Medium | N/A | 3h |
| **TOTAL** | **9** | **Medium** | **Low** | **9h** |

---

## 🔍 Query Examples

### Find delegates with email
```sql
SELECT full_name, badge_name, email
FROM delegates
WHERE email IS NOT NULL AND email != '';
```

### Count delegates by email status
```sql
SELECT 
  COUNT(*) as total,
  SUM(CASE WHEN email IS NOT NULL THEN 1 ELSE 0 END) as with_email,
  SUM(CASE WHEN email IS NULL THEN 1 ELSE 0 END) as without_email
FROM delegates;
```

### Search delegate by email
```sql
SELECT d.*, r.registration_number, r.university_name
FROM delegates d
JOIN registrations r ON d.registration_id = r.id
WHERE d.email LIKE '%@university.edu%';
```

---

## 📧 Future Enhancement Possibilities

Once email field is implemented:

1. **Individual Delegate Notifications**
   ```php
   // Send personalized email to each delegate
   $delegates = getRegistrationDelegates($registrationId);
   foreach ($delegates as $delegate) {
       if ($delegate['email']) {
           sendDelegateEmail($delegate['email'], $delegate);
       }
   }
   ```

2. **Delegate Portal Login**
   ```php
   // Use email for authentication
   SELECT * FROM delegates WHERE email = ?
   ```

3. **Bulk Email by Filter**
   ```php
   // Email all delegates from specific university
   SELECT email FROM delegates d
   JOIN registrations r ON d.registration_id = r.id
   WHERE r.university_name = 'Sunway University'
   AND d.email IS NOT NULL;
   ```

4. **Email Verification**
   ```php
   // Send verification link
   UPDATE delegates 
   SET email_verified = 1 
   WHERE id = ? AND verification_token = ?
   ```

---

This data flow diagram shows the complete journey of the email field through your system, from user input to database storage to admin display! 🚀

