<?php
/**
 * Special Access Links API
 * Handles CRUD operations for special access links
 */

// Start output buffering to catch any unexpected output
ob_start();

require_once 'auth_rbac.php';
require_once '../db_config.php';
// require_once 'log_action.php'; // Removed direct include to avoid method checking

// Provide a safe fallback if the logger helper isn't available
if (!function_exists('logAdminActivity')) {
    function logAdminActivity($pdo, $userId, $action, $entityTable, $entityId, $metaJson = null) {
        // Fallback: write to PHP error log instead of DB
        $summary = sprintf(
            'AdminActivity fallback | user=%s action=%s entity=%s id=%s meta=%s',
            $userId ?? 'unknown',
            $action,
            $entityTable,
            $entityId,
            $metaJson ?? ''
        );
        error_log($summary);
    }
}

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

// Ensure PHP notices/warnings don't break JSON output
ini_set('display_errors', '0');
error_reporting(E_ALL);

// Convert PHP warnings/notices into exceptions so we can return JSON errors
set_error_handler(function ($severity, $message, $file, $line) {
    // Respect @-operator
    if (!(error_reporting() & $severity)) {
        return false;
    }
    throw new ErrorException($message, 0, $severity, $file, $line);
});

// Catch fatal errors and return JSON instead of HTML
register_shutdown_function(function () {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        // Clean any existing output and return JSON error
        if (ob_get_length()) {
            ob_clean();
        }
        http_response_code(500);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'error' => 'Server error: ' . $error['message']
        ]);
    }
});

try {
    // Get database connection
    $pdo = getDBConnection();
    
    // Check authentication and permissions
    $auth = new AdminAuth($pdo);
    $auth->requirePermission('registrations');
    
    // Get current user ID for audit trail
    $userId = $_SESSION['admin_user']['id'] ?? null;
    
    // Handle GET requests (read operations)
    if (strtoupper($_SERVER['REQUEST_METHOD']) === 'GET') {
        $action = $_GET['action'] ?? '';
        
        switch ($action) {
            case 'get':
                // Get single link details
                $linkId = $_GET['id'] ?? null;
                if (!$linkId) {
                    throw new Exception('Link ID is required');
                }
                
                $stmt = $pdo->prepare("
                    SELECT 
                        sal.*,
                        au.full_name as creator_name
                    FROM special_access_links sal
                    LEFT JOIN admin_users au ON sal.created_by = au.id
                    WHERE sal.id = ?
                ");
                $stmt->execute([$linkId]);
                $link = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$link) {
                    throw new Exception('Link not found');
                }
                
                echo json_encode([
                    'success' => true,
                    'data' => $link
                ]);
                exit;
                
            case 'list':
                // Get all links (optional: with filters)
                $stmt = $pdo->query("
                    SELECT 
                        sal.*,
                        au.full_name as creator_name
                    FROM special_access_links sal
                    LEFT JOIN admin_users au ON sal.created_by = au.id
                    ORDER BY sal.created_at DESC
                ");
                $links = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode([
                    'success' => true,
                    'data' => $links
                ]);
                exit;
                
            default:
                throw new Exception('Invalid GET action');
        }
    }
    // Handle POST requests (create, update, delete operations)
    elseif (strtoupper($_SERVER['REQUEST_METHOD']) === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            throw new Exception('Invalid JSON input');
        }
        
        $action = $input['action'] ?? '';
        $data = $input['data'] ?? [];
        
        switch ($action) {
            case 'create':
                // Validate required fields
                if (empty($data['link_name'])) {
                    throw new Exception('Link name is required');
                }
                if (empty($data['link_code'])) {
                    throw new Exception('Link code is required');
                }
                if (empty($data['product_ids']) || !is_array($data['product_ids'])) {
                    throw new Exception('At least one product must be selected');
                }
                
                // Validate link code format
                if (!preg_match('/^[A-Za-z0-9_-]+$/', $data['link_code'])) {
                    throw new Exception('Link code can only contain letters, numbers, dashes, and underscores');
                }
                
                // Check if link code already exists
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM special_access_links WHERE link_code = ?");
                $stmt->execute([$data['link_code']]);
                if ($stmt->fetchColumn() > 0) {
                    throw new Exception('Link code already exists. Please choose a different code.');
                }
                
                // Prepare data
                $productIds = json_encode(array_values($data['product_ids']));
                $includeVisible = isset($data['include_visible_products']) ? intval($data['include_visible_products']) : 1;
                $isActive = isset($data['is_active']) ? intval($data['is_active']) : 1;
                $expiresAt = !empty($data['expires_at']) ? $data['expires_at'] : null;
                
                // Insert new link
                $stmt = $pdo->prepare("
                    INSERT INTO special_access_links 
                    (link_code, link_name, description, custom_message, product_ids, include_visible_products, 
                     is_active, expires_at, created_by, created_at, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
                ");
                
                $customMessage = $data['custom_message'] ?? 'You have exclusive access to additional registration options';
                
                $stmt->execute([
                    $data['link_code'],
                    $data['link_name'],
                    $data['description'] ?? null,
                    $customMessage,
                    $productIds,
                    $includeVisible,
                    $isActive,
                    $expiresAt,
                    $userId
                ]);
                
                $newLinkId = $pdo->lastInsertId();
                
                // Log activity (best-effort)
                if (function_exists('logAdminActivity')) {
                    logAdminActivity($pdo, $userId, 'create_special_link', 'special_access_links', $newLinkId, json_encode([
                        'link_code' => $data['link_code'],
                        'link_name' => $data['link_name'],
                        'product_count' => count($data['product_ids'])
                    ]));
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Special access link created successfully',
                    'link_id' => $newLinkId
                ]);
                exit;
                
            case 'update':
                // Validate required fields
                if (empty($data['link_id'])) {
                    throw new Exception('Link ID is required');
                }
                if (empty($data['link_name'])) {
                    throw new Exception('Link name is required');
                }
                if (empty($data['link_code'])) {
                    throw new Exception('Link code is required');
                }
                if (empty($data['product_ids']) || !is_array($data['product_ids'])) {
                    throw new Exception('At least one product must be selected');
                }
                
                // Validate link code format
                if (!preg_match('/^[A-Za-z0-9_-]+$/', $data['link_code'])) {
                    throw new Exception('Link code can only contain letters, numbers, dashes, and underscores');
                }
                
                // Check if link code already exists (excluding current link)
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM special_access_links WHERE link_code = ? AND id != ?");
                $stmt->execute([$data['link_code'], $data['link_id']]);
                if ($stmt->fetchColumn() > 0) {
                    throw new Exception('Link code already exists. Please choose a different code.');
                }
                
                // Prepare data
                $productIds = json_encode(array_values($data['product_ids']));
                $includeVisible = isset($data['include_visible_products']) ? intval($data['include_visible_products']) : 1;
                $isActive = isset($data['is_active']) ? intval($data['is_active']) : 1;
                $expiresAt = !empty($data['expires_at']) ? $data['expires_at'] : null;
                
                // Update link
                $stmt = $pdo->prepare("
                    UPDATE special_access_links 
                    SET link_code = ?,
                        link_name = ?,
                        description = ?,
                        custom_message = ?,
                        product_ids = ?,
                        include_visible_products = ?,
                        is_active = ?,
                        expires_at = ?,
                        updated_at = NOW()
                    WHERE id = ?
                ");
                
                $customMessage = $data['custom_message'] ?? 'You have exclusive access to additional registration options';
                
                $stmt->execute([
                    $data['link_code'],
                    $data['link_name'],
                    $data['description'] ?? null,
                    $customMessage,
                    $productIds,
                    $includeVisible,
                    $isActive,
                    $expiresAt,
                    $data['link_id']
                ]);
                
                // Log activity (best-effort)
                if (function_exists('logAdminActivity')) {
                    logAdminActivity($pdo, $userId, 'update_special_link', 'special_access_links', $data['link_id'], json_encode([
                        'link_code' => $data['link_code'],
                        'link_name' => $data['link_name'],
                        'product_count' => count($data['product_ids'])
                    ]));
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Special access link updated successfully'
                ]);
                exit;
                
            case 'toggle_status':
                // Toggle active status
                if (empty($data['link_id'])) {
                    throw new Exception('Link ID is required');
                }
                
                $isActive = isset($data['is_active']) ? intval($data['is_active']) : 0;
                
                $stmt = $pdo->prepare("
                    UPDATE special_access_links 
                    SET is_active = ?,
                        updated_at = NOW()
                    WHERE id = ?
                ");
                
                $stmt->execute([$isActive, $data['link_id']]);
                
                // Log activity (best-effort)
                if (function_exists('logAdminActivity')) {
                    logAdminActivity($pdo, $userId, 'toggle_special_link', 'special_access_links', $data['link_id'], json_encode([
                        'is_active' => $isActive
                    ]));
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Link status updated successfully'
                ]);
                exit;
                
            case 'delete':
                // Delete link
                if (empty($data['link_id'])) {
                    throw new Exception('Link ID is required');
                }
                
                // Get link details before deleting (for logging)
                $stmt = $pdo->prepare("SELECT link_code, link_name FROM special_access_links WHERE id = ?");
                $stmt->execute([$data['link_id']]);
                $link = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$link) {
                    throw new Exception('Link not found');
                }
                
                // Delete link
                $stmt = $pdo->prepare("DELETE FROM special_access_links WHERE id = ?");
                $stmt->execute([$data['link_id']]);
                
                // Log activity (best-effort)
                if (function_exists('logAdminActivity')) {
                    logAdminActivity($pdo, $userId, 'delete_special_link', 'special_access_links', $data['link_id'], json_encode([
                        'link_code' => $link['link_code'],
                        'link_name' => $link['link_name']
                    ]));
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Special access link deleted successfully'
                ]);
                exit;
                
            default:
                throw new Exception('Invalid POST action');
        }
    } else {
        throw new Exception('Method not allowed');
    }
    
} catch (Exception $e) {
    // Clear any output buffer
    ob_clean();
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

// End output buffering and ensure clean output
ob_end_flush();


