<?php
require_once 'auth.php';
require_once '../db_config.php';

// Require admin login
requireLogin();

header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    // Load JSON configuration
    $config_file = '../config.json';
    if (!file_exists($config_file)) {
        throw new Exception('Config file not found');
    }
    
    $config = json_decode(file_get_contents($config_file), true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON in config file');
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    // Clear existing settings
    $pdo->exec("DELETE FROM admin_settings");
    
    // Insert settings from JSON
    $stmt = $pdo->prepare("
        INSERT INTO admin_settings (setting_key, setting_value, setting_type, description) 
        VALUES (?, ?, ?, ?)
    ");
    
    // Pricing settings
    $stmt->execute(['earlyBirdActive', $config['pricing']['earlyBirdActive'] ? 'true' : 'false', 'boolean', 'Enable Early Bird pricing']);
    $stmt->execute(['earlyBirdPrice', $config['pricing']['earlyBirdPrice'], 'number', 'Early Bird delegate pass price']);
    $stmt->execute(['standardPrice', $config['pricing']['standardPrice'], 'number', 'Standard delegate pass price']);
    $stmt->execute(['standardTablePrice', $config['pricing']['standardTablePrice'], 'number', 'Standard table presentation price']);
    $stmt->execute(['doubleTablePrice', $config['pricing']['doubleTablePrice'], 'number', 'Double table presentation price']);
    
    // Registration limits
    $stmt->execute(['maxDelegates', $config['registration']['maxDelegates'], 'number', 'Maximum delegates per registration']);
    $stmt->execute(['maxTables', $config['registration']['maxTables'], 'number', 'Maximum table bookings allowed']);
    
    // Event information
    $stmt->execute(['eventName', $config['event']['name'], 'text', 'Event name']);
    $stmt->execute(['eventDates', $config['event']['dates'], 'text', 'Event dates']);
    $stmt->execute(['eventLocation', $config['event']['location'], 'text', 'Event location']);
    
    // Payment information
    $stmt->execute(['bankName', $config['payment']['bankName'], 'text', 'Bank name for transfers']);
    $stmt->execute(['accountName', $config['payment']['accountName'], 'text', 'Bank account name']);
    $stmt->execute(['accountNumber', $config['payment']['accountNumber'], 'text', 'Bank account number']);
    $stmt->execute(['swiftCode', $config['payment']['swiftCode'], 'text', 'Bank SWIFT code']);
    
    // Dietary options (as JSON)
    $stmt->execute(['dietaryOptions', json_encode($config['dietaryOptions']), 'json', 'Available dietary requirement options']);
    
    // Commit transaction
    $pdo->commit();
    
    echo json_encode(['success' => true, 'message' => 'Database settings initialized successfully']);
    
} catch (Exception $e) {
    if (isset($pdo)) {
        $pdo->rollBack();
    }
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
?>
