<?php
require_once '../db_config.php';

// Get configuration from database only (no file fallback)
function getConfiguration() {
    try {
        $pdo = getDBConnection();
        
        // Get all settings from database
        $stmt = $pdo->query("SELECT setting_key, setting_value, setting_type FROM admin_settings");
        $settings = $stmt->fetchAll();
        
        if (empty($settings)) {
            throw new Exception('No configuration found in database. Please run the migration script.');
        }
        
        // Config structure - all values loaded from database
        $config = [
            'pricing' => [],
            'registration' => [],
            'event' => [],
            'payment' => [],
            'dietaryOptions' => [],
            'system' => []
        ];
        
        // Map database settings to config structure
        foreach ($settings as $setting) {
            $key = $setting['setting_key'];
            $value = $setting['setting_value'];
            $type = $setting['setting_type'];
            
            // Convert value based on type
            if ($type === 'boolean') {
                $value = filter_var($value, FILTER_VALIDATE_BOOLEAN);
            } elseif ($type === 'number') {
                $value = is_numeric($value) ? (float)$value : $value;
            } elseif ($type === 'json') {
                $value = json_decode($value, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $value = [];
                }
            }
            
            // Map to config structure
            switch ($key) {
                case 'eventName':
                    $config['event']['name'] = $value;
                    break;
                case 'eventDates':
                    $config['event']['dates'] = $value;
                    break;
                case 'eventLocation':
                    $config['event']['location'] = $value;
                    break;
                case 'bankName':
                    $config['payment']['bankName'] = $value;
                    break;
                case 'accountName':
                    $config['payment']['accountName'] = $value;
                    break;
                case 'accountNumber':
                    $config['payment']['accountNumber'] = $value;
                    break;
                case 'swiftCode':
                    $config['payment']['swiftCode'] = $value;
                    break;
                case 'referencePrefix':
                    $config['payment']['referencePrefix'] = $value;
                    break;
                case 'dietaryOptions':
                    $config['dietaryOptions'] = $value;
                    break;
                case 'systemName':
                    $config['system']['name'] = $value;
                    break;
                case 'adminEmail':
                    $config['system']['adminEmail'] = $value;
                    break;
                case 'timezone':
                    $config['system']['timezone'] = $value;
                    break;
                case 'currency':
                    $config['system']['currency'] = $value;
                    break;
                case 'companyAddress':
                    $config['system']['companyAddress'] = $value;
                    break;
                case 'companyAddress2':
                    $config['system']['companyAddress2'] = $value;
                    break;
                case 'companyPhone':
                    $config['system']['companyPhone'] = $value;
                    break;
                case 'invoiceTermsConditions':
                    $config['system']['invoiceTermsConditions'] = $value;
                    break;
            }
        }
        
        return $config;
        
    } catch (Exception $e) {
        error_log('Configuration error: ' . $e->getMessage());
        throw new Exception('Database configuration not available: ' . $e->getMessage());
    }
}

function getSettingFromDB($pdo, $key, $default = null) {
    $stmt = $pdo->prepare("SELECT setting_value FROM admin_settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $result = $stmt->fetchColumn();
    return $result !== false ? $result : $default;
}
?>
